/**
 * @file
 * Control Geocoder.
 */

(function (Drupal) {

  'use strict';

  /**
   * Geocoder control.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches common map style functionality to relevant elements.
   */
  Drupal.behaviors.geolocationControlGeocoder = {
    attach: function (context, drupalSettings) {

      Drupal.geolocation.executeFeatureOnAllMaps(
        'control_geocoder',

        /**
         * @param {GeolocationMapInterface} map
         * @param {GeolocationMapFeatureSettings} featureSettings
         */
        function (map, featureSettings) {
          Drupal.geolocation.geocoder.addResultCallback(

            /**
             * @param {Object} address.geometry.bounds
             */
            function (address) {
              if (typeof address.geometry.bounds !== 'undefined') {
                map.fitBoundaries(address.geometry.bounds, 'google_control_geocoder');
              }
              else {
                /**
                 * @type {undefined|int}
                 */
                var accuracy = undefined;
                if (typeof address.geometry.accuracy === 'undefined') {
                  accuracy = 10000;
                }
                map.setCenterByCoordinates({lat: address.geometry.location.lat(), lng: address.geometry.location.lng()}, accuracy, 'google_control_geocoder');
              }
            },
            map.id
          );

          return true;
        },
        drupalSettings
      );
    },
    detach: function (context, drupalSettings) {}
  };

})(Drupal);
;
/**
 * @file
 * Marker InfoWindow.
 */

/**
 * @typedef {Object} MarkerInfoWindowSettings
 *
 * @extends {GeolocationMapFeatureSettings}
 *
 * @property {Boolean} infoAutoDisplay
 * @property {Boolean} disableAutoPan
 * @property {Boolean} infoWindowSolitary
 * @property {int} maxWidth
 */

/**
 * @typedef {Object} GoogleInfoWindow
 * @property {Function} open
 * @property {Function} close
 */

/**
 * @property {GoogleInfoWindow} GeolocationGoogleMap.infoWindow
 * @property {function({}):GoogleInfoWindow} GeolocationGoogleMap.InfoWindow
 */

(function (Drupal) {
  "use strict";

  /**
   * Marker InfoWindow.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches common map style functionality to relevant elements.
   */
  Drupal.behaviors.geolocationMarkerInfoWindow = {
    attach: function (context, drupalSettings) {
      Drupal.geolocation.executeFeatureOnAllMaps(
        "marker_infowindow",

        /**
         * @param {GeolocationGoogleMap} map - Current map.
         * @param {MarkerInfoWindowSettings} featureSettings - Settings for current feature.
         */
        function (map, featureSettings) {
          map.addMarkerAddedCallback(function (currentMarker) {
            if (typeof currentMarker.locationWrapper === "undefined") {
              return;
            }

            var content =
              currentMarker.locationWrapper.find(".location-content");

            if (content.length < 1) {
              return;
            }
            content = content.html();

            var markerInfoWindow = {
              content: content.toString(),
              disableAutoPan: featureSettings.disableAutoPan,
            };

            if (featureSettings.maxWidth > 0) {
              markerInfoWindow.maxWidth = featureSettings.maxWidth;
            }

            // Set the info popup text.
            var currentInfoWindow = new google.maps.InfoWindow(
              markerInfoWindow
            );

            currentMarker.addListener("click", function () {
              if (featureSettings.infoWindowSolitary) {
                if (typeof map.infoWindow !== "undefined") {
                  map.infoWindow.close();
                }
                map.infoWindow = currentInfoWindow;
              }
              currentInfoWindow.open(map.googleMap, currentMarker);
            });

            if (featureSettings.infoAutoDisplay) {
              if (map.googleMap.get("tilesloading")) {
                google.maps.event.addListenerOnce(
                  map.googleMap,
                  "tilesloaded",
                  function () {
                    google.maps.event.trigger(currentMarker, "click");
                  }
                );
              } else {
                jQuery.each(map.mapMarkers, function (index, currentMarker) {
                  google.maps.event.trigger(currentMarker, "click");
                });
              }
            }
          });

          return true;
        },
        drupalSettings
      );
    },
    detach: function (context, drupalSettings) {},
  };
})(Drupal);
;
/**
 * @file
 * Control locate.
 */

(function ($, Drupal) {
  "use strict";

  /**
   * Locate control.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches common map style functionality to relevant elements.
   */
  Drupal.behaviors.geolocationControlLocate = {
    attach: function (context, drupalSettings) {
      Drupal.geolocation.executeFeatureOnAllMaps(
        "control_locate",

        /**
         * @param {GeolocationMapInterface} map
         * @param {GeolocationMapFeatureSettings} featureSettings
         */
        function (map, featureSettings) {
          map.addInitializedCallback(function (map) {
            var locateButton = $(
              ".geolocation-map-control .locate",
              map.wrapper
            );

            if (
              navigator.geolocation &&
              window.location.protocol === "https:"
            ) {
              locateButton.click(function (e) {
                navigator.geolocation.getCurrentPosition(function (
                  currentPosition
                ) {
                  var currentLocation = new google.maps.LatLng(
                    currentPosition.coords.latitude,
                    currentPosition.coords.longitude
                  );
                  map.setCenterByCoordinates(
                    currentLocation,
                    currentPosition.coords.accuracy,
                    "google_control_locate"
                  );
                });
                e.preventDefault();
              });
            } else {
              locateButton.remove();
            }
          });

          return true;
        },
        drupalSettings
      );
    },
    detach: function (context, drupalSettings) {},
  };
})(jQuery, Drupal);
;
/**
 * @file
 * Zoom Control.
 */

/**
 * @typedef {Object} ControlZoomSettings
 *
 * @extends {GeolocationMapFeatureSettings}
 *
 * @property {String} behavior
 * @property {String} position
 * @property {String} style
 */

(function (Drupal) {
  "use strict";

  /**
   * Zoom control.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches common map style functionality to relevant elements.
   */
  Drupal.behaviors.geolocationZoomControl = {
    attach: function (context, drupalSettings) {
      Drupal.geolocation.executeFeatureOnAllMaps(
        "control_zoom",

        /**
         * @param {GeolocationGoogleMap} map - Current map.
         * @param {ControlZoomSettings} featureSettings - Settings for current feature.
         */
        function (map, featureSettings) {
          map.addPopulatedCallback(function (map) {
            var options = {
              zoomControlOptions: {
                position: google.maps.ControlPosition[featureSettings.position],
                style: google.maps.ZoomControlStyle[featureSettings.style],
              },
            };

            if (featureSettings.behavior === "always") {
              options.zoomControl = true;
            } else {
              options.zoomControl = undefined;
            }
            map.googleMap.setOptions(options);
          });

          return true;
        },
        drupalSettings
      );
    },
    detach: function (context, drupalSettings) {},
  };
})(Drupal);
;
/**
 * @file
 * Control MapType.
 */

/**
 * @typedef {Object} ControlMapTypeSettings
 *
 * @extends {GeolocationMapFeatureSettings}
 *
 * @property {String} position
 * @property {String} style
 * @property {String} behavior
 */

(function (Drupal) {
  "use strict";

  /**
   * Maptype control.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches common map style functionality to relevant elements.
   */
  Drupal.behaviors.geolocationMapTypeControl = {
    attach: function (context, drupalSettings) {
      Drupal.geolocation.executeFeatureOnAllMaps(
        "control_maptype",

        /**
         * @param {GeolocationGoogleMap} map - Current map.
         * @param {ControlMapTypeSettings} featureSettings - Settings for current feature.
         */
        function (map, featureSettings) {
          map.addPopulatedCallback(function (map) {
            var options = {
              mapTypeControlOptions: {
                position: google.maps.ControlPosition[featureSettings.position],
                style: google.maps.MapTypeControlStyle[featureSettings.style],
              },
            };

            if (featureSettings.behavior === "always") {
              options.mapTypeControl = true;
            } else {
              options.mapTypeControl = undefined;
            }

            map.googleMap.setOptions(options);
          });

          return true;
        },
        drupalSettings
      );
    },
    detach: function (context, drupalSettings) {},
  };
})(Drupal);
;
/**
 * @file
 * Control Recenter.
 */

(function ($, Drupal) {
  "use strict";

  /**
   * Recenter control.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches common map style functionality to relevant elements.
   */
  Drupal.behaviors.geolocationControlRecenter = {
    attach: function (context, drupalSettings) {
      Drupal.geolocation.executeFeatureOnAllMaps(
        "control_recenter",

        /**
         * @param {GeolocationMapInterface} map
         * @param {GeolocationMapFeatureSettings} featureSettings
         */
        function (map, featureSettings) {
          map.addInitializedCallback(function (map) {
            var recenterButton = $(
              ".geolocation-map-control .recenter",
              map.wrapper
            );
            recenterButton.click(function (e) {
              map.setCenter();
              e.preventDefault();
            });
          });

          return true;
        },
        drupalSettings
      );
    },
    detach: function (context, drupalSettings) {},
  };
})(jQuery, Drupal);
;
/**
 * @file
 * Javascript for the plugin-based geocoder function.
 */

/**
 * Callback when map is clicked.
 *
 * @callback GeolocationGeocoderResultCallback
 * @callback GeolocationGeocoderClearCallback
 *
 * @param {Object} address - Address.
 */

/**
 * Geocoder API.
 */
(function ($, Drupal) {
  "use strict";

  Drupal.geolocation = Drupal.geolocation || {};

  Drupal.geolocation.geocoder = Drupal.geolocation.geocoder || {};

  /**
   * Provides the callback that is called when geocoded results are found loads.
   *
   * @param {google.maps.GeocoderResult} result - first returned address
   * @param {string} elementId - Source ID.
   */
  Drupal.geolocation.geocoder.resultCallback = function (result, elementId) {
    Drupal.geolocation.geocoder.resultCallbacks =
      Drupal.geolocation.geocoder.resultCallbacks || [];
    $.each(
      Drupal.geolocation.geocoder.resultCallbacks,
      function (index, callbackContainer) {
        if (callbackContainer.elementId === elementId) {
          callbackContainer.callback(result);
        }
      }
    );
  };

  /**
   * Adds a callback that will be called when results are found.
   *
   * @param {GeolocationGeocoderResultCallback} callback - The callback
   * @param {string} elementId - Identify source of result by its element ID.
   */
  Drupal.geolocation.geocoder.addResultCallback = function (
    callback,
    elementId
  ) {
    if (typeof elementId === "undefined") {
      return;
    }
    Drupal.geolocation.geocoder.resultCallbacks =
      Drupal.geolocation.geocoder.resultCallbacks || [];
    Drupal.geolocation.geocoder.resultCallbacks.push({
      callback: callback,
      elementId: elementId,
    });
  };

  /**
   * Provides the callback that is called when results become invalid loads.
   *
   * @param {string} elementId - Source ID.
   */
  Drupal.geolocation.geocoder.clearCallback = function (elementId) {
    Drupal.geolocation.geocoder.clearCallbacks =
      Drupal.geolocation.geocoder.clearCallbacks || [];
    $.each(
      Drupal.geolocation.geocoder.clearCallbacks,
      function (index, callbackContainer) {
        if (callbackContainer.elementId === elementId) {
          callbackContainer.callback();
        }
      }
    );
  };

  /**
   * Adds a callback that will be called when results should be cleared.
   *
   * @param {GeolocationGeocoderClearCallback} callback - The callback
   * @param {string} elementId - Identify source of result by its element ID.
   */
  Drupal.geolocation.geocoder.addClearCallback = function (
    callback,
    elementId
  ) {
    if (typeof elementId === "undefined") {
      return;
    }
    Drupal.geolocation.geocoder.clearCallbacks =
      Drupal.geolocation.geocoder.clearCallbacks || [];
    Drupal.geolocation.geocoder.clearCallbacks.push({
      callback: callback,
      elementId: elementId,
    });
  };
})(jQuery, Drupal);
;
/**
 * @file
 * Javascript for the Google Geocoding API geocoder.
 */

/**
 * @property {String} drupalSettings.geolocation.geocoder.google_geocoding_api.autocompleteMinLength
 * @property {Object} drupalSettings.geolocation.geocoder.google_geocoding_api.componentRestrictions
 * @property {Object} drupalSettings.geolocation.geocoder.google_geocoding_api.bounds
 * @property {String[]} drupalSettings.geolocation.geocoder.google_geocoding_api.inputIds
 */

(function ($, Drupal) {
  "use strict";

  if (typeof Drupal.geolocation.geocoder === "undefined") {
    return false;
  }

  if (typeof drupalSettings.geolocation === "undefined") {
    return false;
  }

  drupalSettings.geolocation.geocoder.google_geocoding_api =
    drupalSettings.geolocation.geocoder.google_geocoding_api || {};

  Drupal.geolocation.geocoder.googleGeocodingAPI = {};

  var minLength = 1;
  if (
    typeof drupalSettings.geolocation.geocoder.google_geocoding_api
      .autocompleteMinLength !== "undefined" &&
    parseInt(
      drupalSettings.geolocation.geocoder.google_geocoding_api
        .autocompleteMinLength
    )
  ) {
    minLength = parseInt(
      drupalSettings.geolocation.geocoder.google_geocoding_api
        .autocompleteMinLength
    );
  }

  Drupal.geolocation.geocoder.googleGeocodingAPI.attach = function (
    geocoderInput
  ) {
    $(once('geocoder-input', geocoderInput))
      .autocomplete({
        autoFocus: true,
        minLength: minLength,
        source: function (request, response) {
          if (
            typeof Drupal.geolocation.geocoder.googleGeocodingAPI.geocoder ===
            "undefined"
          ) {
            Drupal.geolocation.geocoder.googleGeocodingAPI.geocoder =
              new google.maps.Geocoder();
          }

          var autocompleteResults = [];

          var parameters = {
            address: request.term,
          };
          if (
            typeof drupalSettings.geolocation.geocoder.google_geocoding_api
              .componentRestrictions !== "undefined"
          ) {
            if (
              drupalSettings.geolocation.geocoder.google_geocoding_api
                .componentRestrictions
            ) {
              parameters.componentRestrictions =
                drupalSettings.geolocation.geocoder.google_geocoding_api.componentRestrictions;
            }
          }
          if (
            typeof drupalSettings.geolocation.geocoder.google_geocoding_api
              .bounds !== "undefined"
          ) {
            if (
              drupalSettings.geolocation.geocoder.google_geocoding_api.bounds
            ) {
              parameters.bounds =
                drupalSettings.geolocation.geocoder.google_geocoding_api.bounds;
            }
          }

          Drupal.geolocation.geocoder.googleGeocodingAPI.geocoder.geocode(
            parameters,
            function (results, status) {
              if (status === google.maps.GeocoderStatus.OK) {
                $.each(results, function (index, result) {
                  autocompleteResults.push({
                    value: result.formatted_address,
                    address: result,
                  });
                });
              }
              response(autocompleteResults);
            }
          );
        },

        /**
         * Option form autocomplete selected.
         *
         * @param {Object} event - See jquery doc
         * @param {Object} ui - See jquery doc
         * @param {Object} ui.item - See jquery doc
         */
        select: function (event, ui) {
          if (typeof ui.item.address.geometry.viewport !== "undefined") {
            ui.item.address.geometry.bounds = ui.item.address.geometry.viewport;
          }
          Drupal.geolocation.geocoder.resultCallback(
            ui.item.address,
            $(event.target).data("source-identifier").toString()
          );
        },
      })
      .on("input", function () {
        Drupal.geolocation.geocoder.clearCallback(
          $(this).data("source-identifier").toString()
        );
      });
  };

  /**
   * Attach geocoder input for Google Geocoding API.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches views geocoder input for Google Geocoding API to relevant elements.
   */
  Drupal.behaviors.geolocationGeocoderGoogleGeocodingApi = {
    attach: function (context) {
      Drupal.geolocation.google.addLoadedCallback(function () {
        $.each(
          drupalSettings.geolocation.geocoder.google_geocoding_api.inputIds,
          function (index, inputId) {
            var geocoderInput = $(
              'input.geolocation-geocoder-address[data-source-identifier="' +
                inputId +
                '"]',
              context
            );
            if (geocoderInput.length === 0) {
              return;
            }

            if (geocoderInput.hasClass("geocoder-attached")) {
              return;
            } else {
              geocoderInput.addClass("geocoder-attached");
            }

            if (geocoderInput) {
              Drupal.geolocation.geocoder.googleGeocodingAPI.attach(
                geocoderInput
              );
            }
          }
        );
      });

      // Load Google Maps API and execute all callbacks.
      Drupal.geolocation.google.load();
    },
    detach: function () {},
  };
})(jQuery, Drupal);
;
/**
* DO NOT EDIT THIS FILE.
* See the following change record for more information,
* https://www.drupal.org/node/2815083
* @preserve
**/
(function ($, Drupal) {
  Drupal.user = {
    password: {
      css: {
        passwordParent: 'password-parent',
        passwordsMatch: 'ok',
        passwordsNotMatch: 'error',
        passwordWeak: 'is-weak',
        passwordFair: 'is-fair',
        passwordGood: 'is-good',
        passwordStrong: 'is-strong',
        widgetInitial: '',
        passwordEmpty: '',
        passwordFilled: '',
        confirmEmpty: '',
        confirmFilled: ''
      }
    }
  };
  Drupal.behaviors.password = {
    attach: function attach(context, settings) {
      var cssClasses = Drupal.user.password.css;
      once('password', 'input.js-password-field', context).forEach(function (value) {
        var $mainInput = $(value);
        var $mainInputParent = $mainInput.parent().addClass(cssClasses.passwordParent);
        var $passwordWidget = $mainInput.closest('.js-form-type-password-confirm');
        var $confirmInput = $passwordWidget.find('input.js-password-confirm');
        var $passwordConfirmMessage = $(Drupal.theme('passwordConfirmMessage', settings.password));
        var $passwordMatchStatus = $passwordConfirmMessage.find('[data-drupal-selector="password-match-status-text"]').first();
        if ($passwordMatchStatus.length === 0) {
          $passwordMatchStatus = $passwordConfirmMessage.find('span').first();
          Drupal.deprecationError({
            message: 'Returning <span> without data-drupal-selector="password-match-status-text" attribute is deprecated in drupal:9.1.0 and is removed from drupal:10.0.0. See https://www.drupal.org/node/3152101'
          });
        }
        var $confirmInputParent = $confirmInput.parent().addClass('confirm-parent').append($passwordConfirmMessage);
        var passwordStrengthBarClassesToRemove = [cssClasses.passwordWeak || '', cssClasses.passwordFair || '', cssClasses.passwordGood || '', cssClasses.passwordStrong || ''].join(' ').trim();
        var confirmTextWrapperClassesToRemove = [cssClasses.passwordsMatch || '', cssClasses.passwordsNotMatch || ''].join(' ').trim();
        var widgetClassesToRemove = [cssClasses.widgetInitial || '', cssClasses.passwordEmpty || '', cssClasses.passwordFilled || '', cssClasses.confirmEmpty || '', cssClasses.confirmFilled || ''].join(' ').trim();
        var password = {};
        if (settings.password.showStrengthIndicator) {
          var $passwordStrength = $(Drupal.theme('passwordStrength', settings.password));
          password.$strengthBar = $passwordStrength.find('[data-drupal-selector="password-strength-indicator"]').first();
          if (password.$strengthBar.length === 0) {
            password.$strengthBar = $passwordStrength.find('.js-password-strength__indicator').first();
            Drupal.deprecationError({
              message: 'The js-password-strength__indicator class is deprecated in drupal:9.1.0 and is removed from drupal:10.0.0. Replace js-password-strength__indicator with a data-drupal-selector="password-strength-indicator" attribute. See https://www.drupal.org/node/3152101'
            });
          }
          password.$strengthTextWrapper = $passwordStrength.find('[data-drupal-selector="password-strength-text"]').first();
          if (password.$strengthTextWrapper.length === 0) {
            password.$strengthTextWrapper = $passwordStrength.find('.js-password-strength__text').first();
            Drupal.deprecationError({
              message: 'The js-password-strength__text class is deprecated in drupal:9.1.0 and is removed from drupal:10.0.0. Replace js-password-strength__text with a data-drupal-selector="password-strength-text" attribute. See https://www.drupal.org/node/3152101'
            });
          }
          password.$suggestions = $(Drupal.theme('passwordSuggestions', settings.password, []));
          password.$suggestions.hide();
          $mainInputParent.append($passwordStrength);
          $confirmInputParent.after(password.$suggestions);
        }
        var addWidgetClasses = function addWidgetClasses() {
          $passwordWidget.addClass($mainInput[0].value ? cssClasses.passwordFilled : cssClasses.passwordEmpty).addClass($confirmInput[0].value ? cssClasses.confirmFilled : cssClasses.confirmEmpty);
        };
        var passwordCheckMatch = function passwordCheckMatch(confirmInputVal) {
          var passwordsAreMatching = $mainInput[0].value === confirmInputVal;
          var confirmClass = passwordsAreMatching ? cssClasses.passwordsMatch : cssClasses.passwordsNotMatch;
          var confirmMessage = passwordsAreMatching ? settings.password.confirmSuccess : settings.password.confirmFailure;
          if (!$passwordMatchStatus.hasClass(confirmClass) || !$passwordMatchStatus.html() === confirmMessage) {
            if (confirmTextWrapperClassesToRemove) {
              $passwordMatchStatus.removeClass(confirmTextWrapperClassesToRemove);
            }
            $passwordMatchStatus.html(confirmMessage).addClass(confirmClass);
          }
        };
        var passwordCheck = function passwordCheck() {
          if (settings.password.showStrengthIndicator) {
            var result = Drupal.evaluatePasswordStrength($mainInput[0].value, settings.password);
            var $currentPasswordSuggestions = $(Drupal.theme('passwordSuggestions', settings.password, result.messageTips));
            if (password.$suggestions.html() !== $currentPasswordSuggestions.html()) {
              password.$suggestions.replaceWith($currentPasswordSuggestions);
              password.$suggestions = $currentPasswordSuggestions.toggle(result.strength !== 100);
            }
            if (passwordStrengthBarClassesToRemove) {
              password.$strengthBar.removeClass(passwordStrengthBarClassesToRemove);
            }
            password.$strengthBar.css('width', "".concat(result.strength, "%")).addClass(result.indicatorClass);
            password.$strengthTextWrapper.html(result.indicatorText);
          }
          if ($confirmInput[0].value) {
            passwordCheckMatch($confirmInput[0].value);
            $passwordConfirmMessage.css({
              visibility: 'visible'
            });
          } else {
            $passwordConfirmMessage.css({
              visibility: 'hidden'
            });
          }
          if (widgetClassesToRemove) {
            $passwordWidget.removeClass(widgetClassesToRemove);
            addWidgetClasses();
          }
        };
        if (widgetClassesToRemove) {
          addWidgetClasses();
        }
        $mainInput.on('input', passwordCheck);
        $confirmInput.on('input', passwordCheck);
      });
    }
  };
  Drupal.evaluatePasswordStrength = function (password, passwordSettings) {
    password = password.trim();
    var indicatorText;
    var indicatorClass;
    var weaknesses = 0;
    var strength = 100;
    var msg = [];
    var hasLowercase = /[a-z]/.test(password);
    var hasUppercase = /[A-Z]/.test(password);
    var hasNumbers = /[0-9]/.test(password);
    var hasPunctuation = /[^a-zA-Z0-9]/.test(password);
    var $usernameBox = $('input.username');
    var username = $usernameBox.length > 0 ? $usernameBox[0].value : passwordSettings.username;
    if (password.length < 12) {
      msg.push(passwordSettings.tooShort);
      strength -= (12 - password.length) * 5 + 30;
    }
    if (!hasLowercase) {
      msg.push(passwordSettings.addLowerCase);
      weaknesses += 1;
    }
    if (!hasUppercase) {
      msg.push(passwordSettings.addUpperCase);
      weaknesses += 1;
    }
    if (!hasNumbers) {
      msg.push(passwordSettings.addNumbers);
      weaknesses += 1;
    }
    if (!hasPunctuation) {
      msg.push(passwordSettings.addPunctuation);
      weaknesses += 1;
    }
    switch (weaknesses) {
      case 1:
        strength -= 12.5;
        break;
      case 2:
        strength -= 25;
        break;
      case 3:
      case 4:
        strength -= 40;
        break;
    }
    if (password !== '' && password.toLowerCase() === username.toLowerCase()) {
      msg.push(passwordSettings.sameAsUsername);
      strength = 5;
    }
    var cssClasses = Drupal.user.password.css;
    if (strength < 60) {
      indicatorText = passwordSettings.weak;
      indicatorClass = cssClasses.passwordWeak;
    } else if (strength < 70) {
      indicatorText = passwordSettings.fair;
      indicatorClass = cssClasses.passwordFair;
    } else if (strength < 80) {
      indicatorText = passwordSettings.good;
      indicatorClass = cssClasses.passwordGood;
    } else if (strength <= 100) {
      indicatorText = passwordSettings.strong;
      indicatorClass = cssClasses.passwordStrong;
    }
    var messageTips = msg;
    msg = "".concat(passwordSettings.hasWeaknesses, "<ul><li>").concat(msg.join('</li><li>'), "</li></ul>");
    return Drupal.deprecatedProperty({
      target: {
        strength: strength,
        message: msg,
        indicatorText: indicatorText,
        indicatorClass: indicatorClass,
        messageTips: messageTips
      },
      deprecatedProperty: 'message',
      message: 'The message property is deprecated in drupal:9.1.0 and is removed from drupal:10.0.0. The markup should be constructed using messageTips property and Drupal.theme.passwordSuggestions. See https://www.drupal.org/node/3130352'
    });
  };
})(jQuery, Drupal);;
/**
* DO NOT EDIT THIS FILE.
* See the following change record for more information,
* https://www.drupal.org/node/2815083
* @preserve
**/
(function (Drupal) {
  Drupal.theme.passwordConfirmMessage = function (_ref) {
    var confirmTitle = _ref.confirmTitle;
    var confirmTextWrapper = '<span data-drupal-selector="password-match-status-text"></span>';
    return "<div aria-live=\"polite\" aria-atomic=\"true\" class=\"password-confirm-message\" data-drupal-selector=\"password-confirm-message\">".concat(confirmTitle, " ").concat(confirmTextWrapper, "</div>");
  };
  Drupal.theme.passwordStrength = function (_ref2) {
    var strengthTitle = _ref2.strengthTitle;
    var strengthIndicator = '<div class="password-strength__indicator" data-drupal-selector="password-strength-indicator"></div>';
    var strengthText = '<span class="password-strength__text" data-drupal-selector="password-strength-text"></span>';
    return "\n      <div class=\"password-strength\">\n        <div class=\"password-strength__meter\" data-drupal-selector=\"password-strength-meter\">".concat(strengthIndicator, "</div>\n        <div aria-live=\"polite\" aria-atomic=\"true\" class=\"password-strength__title\">").concat(strengthTitle, " ").concat(strengthText, "</div>\n      </div>\n    ");
  };
  Drupal.theme.passwordSuggestions = function (_ref3, tips) {
    var hasWeaknesses = _ref3.hasWeaknesses;
    return "<div class=\"password-suggestions\">".concat(tips.length ? "".concat(hasWeaknesses, "<ul><li>").concat(tips.join('</li><li>'), "</li></ul>") : '', "</div>");
  };
})(Drupal);;
